if not modules then modules = { } end modules ['char-cjk'] = {
    version   = 1.001,
    comment   = "companion to char-ini.mkiv",
    author    = "Hans Hagen, PRAGMA-ADE, Hasselt NL",
    copyright = "PRAGMA ADE / ConTeXt Development Team",
    license   = "see context related readme files"
}

local setmetatable, next = setmetatable, next
local floor = math.floor
local formatters = string.formatters

local setmetatableindex = table.setmetatableindex

characters     = characters or { }
local ranges   = characters.ranges or { }
local allocate = utilities.storage.allocate

-- Hangul Syllable

do

    -- The following conversion is taken from unicode.org/reports/tr15/tr15-23.html#Hangul
    -- but adapted to our needs.

    -- local SBase = 0xAC00
    --
    -- local LBase, LCount = 0x1100, 19
    -- local VBase, VCount = 0x1161, 21
    -- local TBase, TCount = 0x11A7, 28
    --
    -- local NCount = VCount * TCount
    -- local SCount = LCount * NCount
    --
    -- local function decomposed(unicode)
    --     local SIndex = unicode - SBase
    --     if SIndex >= 0 and SIndex < SCount then
    --         local lead_consonant = LBase + floor( SIndex / NCount)
    --         local medial_vowel   = VBase + floor((SIndex % NCount) / TCount)
    --         local tail_consonant = TBase +        SIndex % TCount
    --         if tail_consonant ~= TBase then
    --             return lead_consonant, medial_vowel, tail_consonant
    --         else
    --             return lead_consonant, medial_vowel
    --         end
    --     end
    -- end
    --
    -- Lua will optimize the inline constants so the next variant is
    -- 10% faster. In practice this will go unnoticed, but it's also less
    -- code, so let's do it. Pushing the constant section into the
    -- function body saves 5%.

    local function decomposed(unicode)
        local index = unicode - 0xAC00
        if index >= 0 and index < 19 * 21 * 28 then
            local lead_consonant = 0x1100 + floor( index / (21 * 28))
            local medial_vowel   = 0x1161 + floor((index % (21 * 28)) / 28)
            local tail_consonant = 0x11A7 +        index % 28
            if tail_consonant ~= 0x11A7 then
                return lead_consonant, medial_vowel, tail_consonant
            else
                return lead_consonant, medial_vowel
            end
        end
    end

    local lead_consonants = { [0] =
        "G", "GG", "N", "D", "DD", "R", "M", "B", "BB",
        "S", "SS", "", "J", "JJ", "C", "K", "T", "P", "H"
    }

    local medial_vowels = { [0] =
        "A", "AE", "YA", "YAE", "EO", "E", "YEO", "YE", "O",
        "WA", "WAE", "OE", "YO", "U", "WEO", "WE", "WI",
        "YU", "EU", "YI", "I"
    }

    local tail_consonants = { [0] =
        "", "G", "GG", "GS", "N", "NJ", "NH", "D", "L", "LG", "LM",
        "LB", "LS", "LT", "LP", "LH", "M", "B", "BS",
        "S", "SS", "NG", "J", "C", "K", "T", "P", "H"
    }

    -- local function description(unicode)
    --     local index = unicode - 0xAC00
    --     if index >= 0 and index < 19 * 21 * 28 then
    --         local lead_consonant = floor( index / NCount)
    --         local medial_vowel   = floor((index % NCount) / TCount)
    --         local tail_consonant =        index % TCount
    --         return formatters["HANGUL SYLLABLE %s%s%s"](
    --             lead_consonants[lead_consonant],
    --             medial_vowels  [medial_vowel  ],
    --             tail_consonants[tail_consonant]
    --         )
    --     end
    -- end

    local function description(unicode)
        local index = unicode - 0xAC00
        if index >= 0 and index < 19 * 21 * 28 then
            local lead_consonant = floor( index / (21 * 28))
            local medial_vowel   = floor((index % (21 * 28)) / 28)
            local tail_consonant =        index % 28
            return formatters["HANGUL SYLLABLE %s%s%s"](
                lead_consonants[lead_consonant],
                medial_vowels  [medial_vowel  ],
                tail_consonants[tail_consonant]
            )
        end
    end

    -- so far

    -- We have a [lead consonant,medial vowel,tail consonant] where the last one
    -- is optional. For sort ranges we need the first one but some are collapsed.
    -- Beware, we map to modern so the font should support it.

    local function leadconsonant(unicode)
        return
     -- unicode  < 0xAC00 and nil       -- original
     -- unicode  > 0xD7AF and nil    or -- original
        unicode >= 0xD558 and 0x314E or -- 하 => ㅎ
        unicode >= 0xD30C and 0x314D or -- 파 => ㅍ
        unicode >= 0xD0C0 and 0x314C or -- 타 => ㅌ
        unicode >= 0xCE74 and 0x314B or -- 카 => ㅋ
        unicode >= 0xCC28 and 0x314A or -- 차 => ㅊ
        unicode >= 0xC790 and 0x3148 or -- 자 => ㅈ
        unicode >= 0xC544 and 0x3147 or -- 아 => ㅇ
        unicode >= 0xC0AC and 0x3145 or -- 사 => ㅅ
        unicode >= 0xBC14 and 0x3142 or -- 바 => ㅂ
        unicode >= 0xB9C8 and 0x3141 or -- 마 => ㅁ
        unicode >= 0xB77C and 0x3139 or -- 라 => ㄹ
        unicode >= 0xB2E4 and 0x3137 or -- 다 => ㄷ
        unicode >= 0xB098 and 0x3134 or -- 나 => ㄴ
        unicode >= 0xAC00 and 0x3131 or -- 가 => ㄱ
                              nil       -- can't happen
    end

    local remapped = { -- this might be merged into char-def.lua
        [0x1100] = 0x3131, -- G
        [0x1101] = 0x3132, -- GG
        [0x1102] = 0x3134, -- N
        [0x1103] = 0x3137, -- D
        [0x1104] = 0x3138, -- DD
        [0x1105] = 0x3139, -- R
     -- [0X111A] = 0x3140, -- LH used for last sound
        [0x1106] = 0x3141, -- M
        [0x1107] = 0x3142, -- B
        [0x1108] = 0x3143, -- BB
     -- [0x1121] = 0x3144, -- BS used for last sound
        [0x1109] = 0x3145, -- S
        [0x110A] = 0x3146, -- SS
        [0x110B] = 0x3147, -- (IEUNG) no sound but has form
        [0x110C] = 0x3148, -- J
        [0x110D] = 0x3149, -- JJ
        [0x110E] = 0x314A, -- C
        [0x110F] = 0x314B, -- K
        [0x1110] = 0x314C, -- T
        [0x1111] = 0x314D, -- P
        [0x1112] = 0x314E, -- H

        [0x1161] = 0x314F, -- A
        [0x1162] = 0x3150, -- AE
        [0x1163] = 0x3151, -- YA
        [0x1164] = 0x3152, -- YAE
        [0x1165] = 0x3153, -- EO
        [0x1166] = 0x3154, -- E
        [0x1167] = 0x3155, -- YEO
        [0x1168] = 0x3156, -- YE
        [0x1169] = 0x3157, -- O
        [0x116A] = 0x3158, -- WA
        [0x116B] = 0x3159, -- WAE
        [0x116C] = 0x315A, -- OE
        [0x116D] = 0x315B, -- YO
        [0x116E] = 0x315C, -- U
        [0x116F] = 0x315D, -- WEO
        [0x1170] = 0x315E, -- WE
        [0x1171] = 0x315F, -- WI
        [0x1172] = 0x3160, -- YU
        [0x1173] = 0x3161, -- EU
        [0x1174] = 0x3162, -- YI
        [0x1175] = 0x3163, -- I

        [0x11A8] = 0x3131, -- G
        [0x11A9] = 0x3132, -- GG
        [0x11AA] = 0x3133, -- GS
        [0x11AB] = 0x3134, -- N
        [0x11AC] = 0x3135, -- NJ
        [0x11AD] = 0x3136, -- NH
        [0x11AE] = 0x3137, -- D
        [0x11AF] = 0x3139, -- L
        [0x11B0] = 0x313A, -- LG
        [0x11B1] = 0x313B, -- LM
        [0x11B2] = 0x313C, -- LB
        [0x11B3] = 0x313D, -- LS
        [0x11B4] = 0x313E, -- LT
        [0x11B5] = 0x313F, -- LP
        [0x11B6] = 0x3140, -- LH
        [0x11B7] = 0x3141, -- M
        [0x11B8] = 0x3142, -- B
        [0x11B9] = 0x3144, -- BS
        [0x11BA] = 0x3145, -- S
        [0x11BB] = 0x3146, -- SS
        [0x11BC] = 0x3147, -- NG
        [0x11BD] = 0x3148, -- J
        [0x11BE] = 0x314A, -- C
        [0x11BF] = 0x314B, -- K
        [0x11C0] = 0x314C, -- T
        [0x11C1] = 0x314D, -- P
        [0x11C2] = 0x314E, -- H
    }

    characters.hangul = allocate {
        decomposed    = decomposed,
        description   = description,
        leadconsonant = leadconsonant,
        remapped      = remapped,
    }

    function characters.remap_hangul_syllabe(t)
        local tt = type(t)
        if tt == "number" then
            return remapped[t] or t
        elseif tt == "table" then
            local r = { }
            for i=1,#t do
                local ti = t[i]
                r[i] = remapped[ti] or ti
            end
            return r
        else
            return t
        end
    end

    -- so far

    local common = {
        category    = "lo",
        cjkwd       = "w",
        description = "<Hangul Syllable>",
        direction   = "l",
        linebreak   = "h2",
    }

    local metatable = {
        __index = function(t,k)
            local u = t.unicodeslot
            if k == "fscode" or k == "leadconsonant" then
                return leadconsonant(u)
            elseif k == "decomposed" then
                return { decomposed(u) }
            elseif k == "specials" then
                return { "char", decomposed(u) }
            elseif k == "description" then
                return description(u)
            else
                return common[k]
            end
        end
    }

    local function extender(k)
        local t = {
            unicodeslot = k,
        }
        setmetatable(t,metatable)
        return t
    end

    local range = {
        name     = "hangul",
        first    = 0xAC00,
        last     = 0xD7A3,
        extender = extender,
    }

    setmetatable(range,metatable) -- hm

    ranges[#ranges+1] = range

end

-- CJK Ideograph : we need to check it

do

    local knownvariants = {
        [0x0349E] = { [0x0FE00] = 0x2F80C },
        [0x034B9] = { [0x0FE00] = 0x2F813 },
        [0x034BB] = { [0x0FE00] = 0x2F9CA },
        [0x034DF] = { [0x0FE00] = 0x2F81F },
        [0x03515] = { [0x0FE00] = 0x2F824 },
        [0x036EE] = { [0x0FE00] = 0x2F867 },
        [0x036FC] = { [0x0FE00] = 0x2F868 },
        [0x03781] = { [0x0FE00] = 0x2F876 },
        [0x0382F] = { [0x0FE00] = 0x2F883 },
        [0x03862] = { [0x0FE00] = 0x2F888 },
        [0x0387C] = { [0x0FE00] = 0x2F88A },
        [0x038C7] = { [0x0FE00] = 0x2F896 },
        [0x038E3] = { [0x0FE00] = 0x2F89B },
        [0x0391C] = { [0x0FE00] = 0x2F8A2 },
        [0x0393A] = { [0x0FE00] = 0x2F8A1 },
        [0x03A2E] = { [0x0FE00] = 0x2F8C2 },
        [0x03A6C] = { [0x0FE00] = 0x2F8C7 },
        [0x03AE4] = { [0x0FE00] = 0x2F8D1 },
        [0x03B08] = { [0x0FE00] = 0x2F8D0 },
        [0x03B19] = { [0x0FE00] = 0x2F8CE },
        [0x03B49] = { [0x0FE00] = 0x2F8DE },
        [0x03B9D] = { [0x0FE00] = 0x0FAD2, [0x0FE01] = 0x2F8E7 },
        [0x03C18] = { [0x0FE00] = 0x2F8EE },
        [0x03C4E] = { [0x0FE00] = 0x2F8F2 },
        [0x03D33] = { [0x0FE00] = 0x2F90A },
        [0x03D96] = { [0x0FE00] = 0x2F916 },
        [0x03EAC] = { [0x0FE00] = 0x2F92A },
        [0x03EB8] = { [0x0FE00] = 0x2F92C, [0x0FE01] = 0x2F92D },
        [0x03F1B] = { [0x0FE00] = 0x2F933 },
        [0x03FFC] = { [0x0FE00] = 0x2F93E },
        [0x04008] = { [0x0FE00] = 0x2F93F },
        [0x04018] = { [0x0FE00] = 0x0FAD3 },
        [0x04039] = { [0x0FE00] = 0x0FAD4, [0x0FE01] = 0x2F949 },
        [0x04046] = { [0x0FE00] = 0x2F94B },
        [0x04096] = { [0x0FE00] = 0x2F94C },
        [0x040E3] = { [0x0FE00] = 0x2F951 },
        [0x0412F] = { [0x0FE00] = 0x2F958 },
        [0x04202] = { [0x0FE00] = 0x2F960 },
        [0x04227] = { [0x0FE00] = 0x2F964 },
        [0x042A0] = { [0x0FE00] = 0x2F967 },
        [0x04301] = { [0x0FE00] = 0x2F96D },
        [0x04334] = { [0x0FE00] = 0x2F971 },
        [0x04359] = { [0x0FE00] = 0x2F974 },
        [0x043D5] = { [0x0FE00] = 0x2F981 },
        [0x043D9] = { [0x0FE00] = 0x2F8D7 },
        [0x0440B] = { [0x0FE00] = 0x2F984 },
        [0x0446B] = { [0x0FE00] = 0x2F98E },
        [0x0452B] = { [0x0FE00] = 0x2F9A7 },
        [0x0455D] = { [0x0FE00] = 0x2F9AE },
        [0x04561] = { [0x0FE00] = 0x2F9AF },
        [0x0456B] = { [0x0FE00] = 0x2F9B2 },
        [0x045D7] = { [0x0FE00] = 0x2F9BF },
        [0x045F9] = { [0x0FE00] = 0x2F9C2 },
        [0x04635] = { [0x0FE00] = 0x2F9C8 },
        [0x046BE] = { [0x0FE00] = 0x2F9CD },
        [0x046C7] = { [0x0FE00] = 0x2F9CE },
        [0x04995] = { [0x0FE00] = 0x2F9EF },
        [0x049E6] = { [0x0FE00] = 0x2F9F2 },
        [0x04A6E] = { [0x0FE00] = 0x2F9F8 },
        [0x04A76] = { [0x0FE00] = 0x2F9F9 },
        [0x04AB2] = { [0x0FE00] = 0x2F9FC },
        [0x04B33] = { [0x0FE00] = 0x2FA03 },
        [0x04BCE] = { [0x0FE00] = 0x2FA08 },
        [0x04CCE] = { [0x0FE00] = 0x2FA0D },
        [0x04CED] = { [0x0FE00] = 0x2FA0E },
        [0x04CF8] = { [0x0FE00] = 0x2FA11 },
        [0x04D56] = { [0x0FE00] = 0x2FA16 },
        [0x04E0D] = { [0x0FE00] = 0x0F967 },
        [0x04E26] = { [0x0FE00] = 0x0FA70 },
        [0x04E32] = { [0x0FE00] = 0x0F905 },
        [0x04E38] = { [0x0FE00] = 0x2F801 },
        [0x04E39] = { [0x0FE00] = 0x0F95E },
        [0x04E3D] = { [0x0FE00] = 0x2F800 },
        [0x04E41] = { [0x0FE00] = 0x2F802 },
        [0x04E82] = { [0x0FE00] = 0x0F91B },
        [0x04E86] = { [0x0FE00] = 0x0F9BA },
        [0x04EAE] = { [0x0FE00] = 0x0F977 },
        [0x04EC0] = { [0x0FE00] = 0x0F9FD },
        [0x04ECC] = { [0x0FE00] = 0x2F819 },
        [0x04EE4] = { [0x0FE00] = 0x0F9A8 },
        [0x04F60] = { [0x0FE00] = 0x2F804 },
        [0x04F80] = { [0x0FE00] = 0x0FA73 },
        [0x04F86] = { [0x0FE00] = 0x0F92D },
        [0x04F8B] = { [0x0FE00] = 0x0F9B5 },
        [0x04FAE] = { [0x0FE00] = 0x0FA30, [0x0FE01] = 0x2F805 },
        [0x04FBB] = { [0x0FE00] = 0x2F806 },
        [0x04FBF] = { [0x0FE00] = 0x0F965 },
        [0x05002] = { [0x0FE00] = 0x2F807 },
        [0x0502B] = { [0x0FE00] = 0x0F9D4 },
        [0x0507A] = { [0x0FE00] = 0x2F808 },
        [0x05099] = { [0x0FE00] = 0x2F809 },
        [0x050CF] = { [0x0FE00] = 0x2F80B },
        [0x050DA] = { [0x0FE00] = 0x0F9BB },
        [0x050E7] = { [0x0FE00] = 0x0FA31, [0x0FE01] = 0x2F80A },
        [0x05140] = { [0x0FE00] = 0x0FA0C },
        [0x05145] = { [0x0FE00] = 0x0FA74 },
        [0x0514D] = { [0x0FE00] = 0x0FA32, [0x0FE01] = 0x2F80E },
        [0x05154] = { [0x0FE00] = 0x2F80F },
        [0x05164] = { [0x0FE00] = 0x2F810 },
        [0x05167] = { [0x0FE00] = 0x2F814 },
        [0x05168] = { [0x0FE00] = 0x0FA72 },
        [0x05169] = { [0x0FE00] = 0x0F978 },
        [0x0516D] = { [0x0FE00] = 0x0F9D1 },
        [0x05177] = { [0x0FE00] = 0x2F811 },
        [0x05180] = { [0x0FE00] = 0x0FA75 },
        [0x0518D] = { [0x0FE00] = 0x2F815 },
        [0x05192] = { [0x0FE00] = 0x2F8D2 },
        [0x05195] = { [0x0FE00] = 0x2F8D3 },
        [0x05197] = { [0x0FE00] = 0x2F817 },
        [0x051A4] = { [0x0FE00] = 0x2F818 },
        [0x051AC] = { [0x0FE00] = 0x2F81A },
        [0x051B5] = { [0x0FE00] = 0x0FA71, [0x0FE01] = 0x2F81B },
        [0x051B7] = { [0x0FE00] = 0x0F92E },
        [0x051C9] = { [0x0FE00] = 0x0F979 },
        [0x051CC] = { [0x0FE00] = 0x0F955 },
        [0x051DC] = { [0x0FE00] = 0x0F954 },
        [0x051DE] = { [0x0FE00] = 0x0FA15 },
        [0x051F5] = { [0x0FE00] = 0x2F81D },
        [0x05203] = { [0x0FE00] = 0x2F81E },
        [0x05207] = { [0x0FE00] = 0x0FA00, [0x0FE01] = 0x2F850 },
        [0x05217] = { [0x0FE00] = 0x0F99C },
        [0x05229] = { [0x0FE00] = 0x0F9DD },
        [0x0523A] = { [0x0FE00] = 0x0F9FF },
        [0x0523B] = { [0x0FE00] = 0x2F820 },
        [0x05246] = { [0x0FE00] = 0x2F821 },
        [0x05272] = { [0x0FE00] = 0x2F822 },
        [0x05277] = { [0x0FE00] = 0x2F823 },
        [0x05289] = { [0x0FE00] = 0x0F9C7 },
        [0x0529B] = { [0x0FE00] = 0x0F98A },
        [0x052A3] = { [0x0FE00] = 0x0F99D },
        [0x052B3] = { [0x0FE00] = 0x2F992 },
        [0x052C7] = { [0x0FE00] = 0x0FA76, [0x0FE01] = 0x2F825 },
        [0x052C9] = { [0x0FE00] = 0x0FA33, [0x0FE01] = 0x2F826 },
        [0x052D2] = { [0x0FE00] = 0x0F952 },
        [0x052DE] = { [0x0FE00] = 0x0F92F },
        [0x052E4] = { [0x0FE00] = 0x0FA34, [0x0FE01] = 0x2F827 },
        [0x052F5] = { [0x0FE00] = 0x0F97F },
        [0x052FA] = { [0x0FE00] = 0x0FA77, [0x0FE01] = 0x2F828 },
        [0x05305] = { [0x0FE00] = 0x2F829 },
        [0x05306] = { [0x0FE00] = 0x2F82A },
        [0x05317] = { [0x0FE00] = 0x0F963, [0x0FE01] = 0x2F82B },
        [0x0533F] = { [0x0FE00] = 0x0F9EB },
        [0x05349] = { [0x0FE00] = 0x2F82C },
        [0x05351] = { [0x0FE00] = 0x0FA35, [0x0FE01] = 0x2F82D },
        [0x0535A] = { [0x0FE00] = 0x2F82E },
        [0x05373] = { [0x0FE00] = 0x2F82F },
        [0x05375] = { [0x0FE00] = 0x0F91C },
        [0x0537D] = { [0x0FE00] = 0x2F830 },
        [0x0537F] = { [0x0FE00] = 0x2F831, [0x0FE01] = 0x2F832, [0x0FE02] = 0x2F833 },
        [0x053C3] = { [0x0FE00] = 0x0F96B },
        [0x053CA] = { [0x0FE00] = 0x2F836 },
        [0x053DF] = { [0x0FE00] = 0x2F837 },
        [0x053E5] = { [0x0FE00] = 0x0F906 },
        [0x053EB] = { [0x0FE00] = 0x2F839 },
        [0x053F1] = { [0x0FE00] = 0x2F83A },
        [0x05406] = { [0x0FE00] = 0x2F83B },
        [0x0540F] = { [0x0FE00] = 0x0F9DE },
        [0x0541D] = { [0x0FE00] = 0x0F9ED },
        [0x05438] = { [0x0FE00] = 0x2F83D },
        [0x05442] = { [0x0FE00] = 0x0F980 },
        [0x05448] = { [0x0FE00] = 0x2F83E },
        [0x05468] = { [0x0FE00] = 0x2F83F },
        [0x0549E] = { [0x0FE00] = 0x2F83C },
        [0x054A2] = { [0x0FE00] = 0x2F840 },
        [0x054BD] = { [0x0FE00] = 0x0F99E },
        [0x054F6] = { [0x0FE00] = 0x2F841 },
        [0x05510] = { [0x0FE00] = 0x2F842 },
        [0x05553] = { [0x0FE00] = 0x2F843 },
        [0x05555] = { [0x0FE00] = 0x0FA79 },
        [0x05563] = { [0x0FE00] = 0x2F844 },
        [0x05584] = { [0x0FE00] = 0x2F845, [0x0FE01] = 0x2F846 },
        [0x05587] = { [0x0FE00] = 0x0F90B },
        [0x05599] = { [0x0FE00] = 0x0FA7A, [0x0FE01] = 0x2F847 },
        [0x0559D] = { [0x0FE00] = 0x0FA36, [0x0FE01] = 0x0FA78 },
        [0x055AB] = { [0x0FE00] = 0x2F848 },
        [0x055B3] = { [0x0FE00] = 0x2F849 },
        [0x055C0] = { [0x0FE00] = 0x0FA0D },
        [0x055C2] = { [0x0FE00] = 0x2F84A },
        [0x055E2] = { [0x0FE00] = 0x0FA7B },
        [0x05606] = { [0x0FE00] = 0x0FA37, [0x0FE01] = 0x2F84C },
        [0x05651] = { [0x0FE00] = 0x2F84E },
        [0x05668] = { [0x0FE00] = 0x0FA38 },
        [0x05674] = { [0x0FE00] = 0x2F84F },
        [0x056F9] = { [0x0FE00] = 0x0F9A9 },
        [0x05716] = { [0x0FE00] = 0x2F84B },
        [0x05717] = { [0x0FE00] = 0x2F84D },
        [0x0578B] = { [0x0FE00] = 0x2F855 },
        [0x057CE] = { [0x0FE00] = 0x2F852 },
        [0x057F4] = { [0x0FE00] = 0x2F853 },
        [0x0580D] = { [0x0FE00] = 0x2F854 },
        [0x05831] = { [0x0FE00] = 0x2F857 },
        [0x05832] = { [0x0FE00] = 0x2F856 },
        [0x05840] = { [0x0FE00] = 0x0FA39 },
        [0x0585A] = { [0x0FE00] = 0x0FA10, [0x0FE01] = 0x0FA7C },
        [0x0585E] = { [0x0FE00] = 0x0F96C },
        [0x058A8] = { [0x0FE00] = 0x0FA3A },
        [0x058AC] = { [0x0FE00] = 0x2F858 },
        [0x058B3] = { [0x0FE00] = 0x0FA7D },
        [0x058D8] = { [0x0FE00] = 0x0F94A },
        [0x058DF] = { [0x0FE00] = 0x0F942 },
        [0x058EE] = { [0x0FE00] = 0x2F851 },
        [0x058F2] = { [0x0FE00] = 0x2F85A },
        [0x058F7] = { [0x0FE00] = 0x2F85B },
        [0x05906] = { [0x0FE00] = 0x2F85C },
        [0x0591A] = { [0x0FE00] = 0x2F85D },
        [0x05922] = { [0x0FE00] = 0x2F85E },
        [0x05944] = { [0x0FE00] = 0x0FA7E },
        [0x05948] = { [0x0FE00] = 0x0F90C },
        [0x05951] = { [0x0FE00] = 0x0F909 },
        [0x05954] = { [0x0FE00] = 0x0FA7F },
        [0x05962] = { [0x0FE00] = 0x2F85F },
        [0x05973] = { [0x0FE00] = 0x0F981 },
        [0x059D8] = { [0x0FE00] = 0x2F865 },
        [0x059EC] = { [0x0FE00] = 0x2F862 },
        [0x05A1B] = { [0x0FE00] = 0x2F863 },
        [0x05A27] = { [0x0FE00] = 0x2F864 },
        [0x05A62] = { [0x0FE00] = 0x0FA80 },
        [0x05A66] = { [0x0FE00] = 0x2F866 },
        [0x05AB5] = { [0x0FE00] = 0x2F986 },
        [0x05B08] = { [0x0FE00] = 0x2F869 },
        [0x05B28] = { [0x0FE00] = 0x0FA81 },
        [0x05B3E] = { [0x0FE00] = 0x2F86A, [0x0FE01] = 0x2F86B },
        [0x05B85] = { [0x0FE00] = 0x0FA04 },
        [0x05BC3] = { [0x0FE00] = 0x2F86D },
        [0x05BD8] = { [0x0FE00] = 0x2F86E },
        [0x05BE7] = { [0x0FE00] = 0x0F95F, [0x0FE01] = 0x0F9AA, [0x0FE02] = 0x2F86F },
        [0x05BEE] = { [0x0FE00] = 0x0F9BC },
        [0x05BF3] = { [0x0FE00] = 0x2F870 },
        [0x05BFF] = { [0x0FE00] = 0x2F872 },
        [0x05C06] = { [0x0FE00] = 0x2F873 },
        [0x05C22] = { [0x0FE00] = 0x2F875 },
        [0x05C3F] = { [0x0FE00] = 0x0F9BD },
        [0x05C60] = { [0x0FE00] = 0x2F877 },
        [0x05C62] = { [0x0FE00] = 0x0F94B },
        [0x05C64] = { [0x0FE00] = 0x0FA3B },
        [0x05C65] = { [0x0FE00] = 0x0F9DF },
        [0x05C6E] = { [0x0FE00] = 0x0FA3C, [0x0FE01] = 0x2F878 },
        [0x05C8D] = { [0x0FE00] = 0x2F87A },
        [0x05CC0] = { [0x0FE00] = 0x2F879 },
        [0x05D19] = { [0x0FE00] = 0x0F9D5 },
        [0x05D43] = { [0x0FE00] = 0x2F87C },
        [0x05D50] = { [0x0FE00] = 0x0F921 },
        [0x05D6B] = { [0x0FE00] = 0x2F87F },
        [0x05D6E] = { [0x0FE00] = 0x2F87E },
        [0x05D7C] = { [0x0FE00] = 0x2F880 },
        [0x05DB2] = { [0x0FE00] = 0x2F9F4 },
        [0x05DBA] = { [0x0FE00] = 0x0F9AB },
        [0x05DE1] = { [0x0FE00] = 0x2F881 },
        [0x05DE2] = { [0x0FE00] = 0x2F882 },
        [0x05DFD] = { [0x0FE00] = 0x2F884 },
        [0x05E28] = { [0x0FE00] = 0x2F885 },
        [0x05E3D] = { [0x0FE00] = 0x2F886 },
        [0x05E69] = { [0x0FE00] = 0x2F887 },
        [0x05E74] = { [0x0FE00] = 0x0F98E },
        [0x05EA6] = { [0x0FE00] = 0x0FA01 },
        [0x05EB0] = { [0x0FE00] = 0x2F88B },
        [0x05EB3] = { [0x0FE00] = 0x2F88C },
        [0x05EB6] = { [0x0FE00] = 0x2F88D },
        [0x05EC9] = { [0x0FE00] = 0x0F9A2 },
        [0x05ECA] = { [0x0FE00] = 0x0F928, [0x0FE01] = 0x2F88E },
        [0x05ED2] = { [0x0FE00] = 0x0FA82 },
        [0x05ED3] = { [0x0FE00] = 0x0FA0B },
        [0x05ED9] = { [0x0FE00] = 0x0FA83 },
        [0x05EEC] = { [0x0FE00] = 0x0F982 },
        [0x05EFE] = { [0x0FE00] = 0x2F890 },
        [0x05F04] = { [0x0FE00] = 0x0F943 },
        [0x05F22] = { [0x0FE00] = 0x2F894, [0x0FE01] = 0x2F895 },
        [0x05F53] = { [0x0FE00] = 0x2F874 },
        [0x05F62] = { [0x0FE00] = 0x2F899 },
        [0x05F69] = { [0x0FE00] = 0x0FA84 },
        [0x05F6B] = { [0x0FE00] = 0x2F89A },
        [0x05F8B] = { [0x0FE00] = 0x0F9D8 },
        [0x05F9A] = { [0x0FE00] = 0x2F89C },
        [0x05FA9] = { [0x0FE00] = 0x0F966 },
        [0x05FAD] = { [0x0FE00] = 0x0FA85 },
        [0x05FCD] = { [0x0FE00] = 0x2F89D },
        [0x05FD7] = { [0x0FE00] = 0x2F89E },
        [0x05FF5] = { [0x0FE00] = 0x0F9A3 },
        [0x05FF9] = { [0x0FE00] = 0x2F89F },
        [0x06012] = { [0x0FE00] = 0x0F960 },
        [0x0601C] = { [0x0FE00] = 0x0F9AC },
        [0x06075] = { [0x0FE00] = 0x0FA6B },
        [0x06081] = { [0x0FE00] = 0x2F8A0 },
        [0x06094] = { [0x0FE00] = 0x0FA3D, [0x0FE01] = 0x2F8A3 },
        [0x060C7] = { [0x0FE00] = 0x2F8A5 },
        [0x060D8] = { [0x0FE00] = 0x0FA86 },
        [0x060E1] = { [0x0FE00] = 0x0F9B9 },
        [0x06108] = { [0x0FE00] = 0x0FA88 },
        [0x06144] = { [0x0FE00] = 0x0F9D9 },
        [0x06148] = { [0x0FE00] = 0x2F8A6 },
        [0x0614C] = { [0x0FE00] = 0x2F8A7, [0x0FE01] = 0x2F8A9 },
        [0x0614E] = { [0x0FE00] = 0x0FA87, [0x0FE01] = 0x2F8A8 },
        [0x06160] = { [0x0FE00] = 0x0FA8A },
        [0x06168] = { [0x0FE00] = 0x0FA3E },
        [0x0617A] = { [0x0FE00] = 0x2F8AA },
        [0x0618E] = { [0x0FE00] = 0x0FA3F, [0x0FE01] = 0x0FA89, [0x0FE02] = 0x2F8AB },
        [0x06190] = { [0x0FE00] = 0x0F98F },
        [0x061A4] = { [0x0FE00] = 0x2F8AD },
        [0x061AF] = { [0x0FE00] = 0x2F8AE },
        [0x061B2] = { [0x0FE00] = 0x2F8AC },
        [0x061DE] = { [0x0FE00] = 0x2F8AF },
        [0x061F2] = { [0x0FE00] = 0x0FA40, [0x0FE01] = 0x0FA8B, [0x0FE02] = 0x2F8B0 },
        [0x061F6] = { [0x0FE00] = 0x0F90D, [0x0FE01] = 0x2F8B1 },
        [0x06200] = { [0x0FE00] = 0x0F990 },
        [0x06210] = { [0x0FE00] = 0x2F8B2 },
        [0x0621B] = { [0x0FE00] = 0x2F8B3 },
        [0x0622E] = { [0x0FE00] = 0x0F9D2 },
        [0x06234] = { [0x0FE00] = 0x0FA8C },
        [0x0625D] = { [0x0FE00] = 0x2F8B4 },
        [0x062B1] = { [0x0FE00] = 0x2F8B5 },
        [0x062C9] = { [0x0FE00] = 0x0F925 },
        [0x062CF] = { [0x0FE00] = 0x0F95B },
        [0x062D3] = { [0x0FE00] = 0x0FA02 },
        [0x062D4] = { [0x0FE00] = 0x2F8B6 },
        [0x062FC] = { [0x0FE00] = 0x2F8BA },
        [0x062FE] = { [0x0FE00] = 0x0F973 },
        [0x0633D] = { [0x0FE00] = 0x2F8B9 },
        [0x06350] = { [0x0FE00] = 0x2F8B7 },
        [0x06368] = { [0x0FE00] = 0x2F8BB },
        [0x0637B] = { [0x0FE00] = 0x0F9A4 },
        [0x06383] = { [0x0FE00] = 0x2F8BC },
        [0x063A0] = { [0x0FE00] = 0x0F975 },
        [0x063A9] = { [0x0FE00] = 0x2F8C1 },
        [0x063C4] = { [0x0FE00] = 0x0FA8D },
        [0x063C5] = { [0x0FE00] = 0x2F8C0 },
        [0x063E4] = { [0x0FE00] = 0x2F8BD },
        [0x0641C] = { [0x0FE00] = 0x0FA8E },
        [0x06422] = { [0x0FE00] = 0x2F8BF },
        [0x06452] = { [0x0FE00] = 0x0FA8F },
        [0x06469] = { [0x0FE00] = 0x2F8C3 },
        [0x06477] = { [0x0FE00] = 0x2F8C6 },
        [0x0647E] = { [0x0FE00] = 0x2F8C4 },
        [0x0649A] = { [0x0FE00] = 0x0F991 },
        [0x0649D] = { [0x0FE00] = 0x2F8C5 },
        [0x064C4] = { [0x0FE00] = 0x0F930 },
        [0x0654F] = { [0x0FE00] = 0x0FA41, [0x0FE01] = 0x2F8C8 },
        [0x06556] = { [0x0FE00] = 0x0FA90 },
        [0x0656C] = { [0x0FE00] = 0x2F8C9 },
        [0x06578] = { [0x0FE00] = 0x0F969 },
        [0x06599] = { [0x0FE00] = 0x0F9BE },
        [0x065C5] = { [0x0FE00] = 0x0F983 },
        [0x065E2] = { [0x0FE00] = 0x0FA42 },
        [0x065E3] = { [0x0FE00] = 0x2F8CB },
        [0x06613] = { [0x0FE00] = 0x0F9E0 },
        [0x06649] = { [0x0FE00] = 0x2F8CD },
        [0x06674] = { [0x0FE00] = 0x0FA12, [0x0FE01] = 0x0FA91 },
        [0x06688] = { [0x0FE00] = 0x0F9C5 },
        [0x06691] = { [0x0FE00] = 0x0FA43, [0x0FE01] = 0x2F8CF },
        [0x0669C] = { [0x0FE00] = 0x2F8D5 },
        [0x066B4] = { [0x0FE00] = 0x0FA06 },
        [0x066C6] = { [0x0FE00] = 0x0F98B },
        [0x066F4] = { [0x0FE00] = 0x0F901 },
        [0x066F8] = { [0x0FE00] = 0x2F8CC },
        [0x06700] = { [0x0FE00] = 0x2F8D4 },
        [0x06717] = { [0x0FE00] = 0x0F929, [0x0FE01] = 0x0FA92, [0x0FE02] = 0x2F8D8 },
        [0x0671B] = { [0x0FE00] = 0x0FA93, [0x0FE01] = 0x2F8D9 },
        [0x06721] = { [0x0FE00] = 0x2F8DA },
        [0x0674E] = { [0x0FE00] = 0x0F9E1 },
        [0x06753] = { [0x0FE00] = 0x2F8DC },
        [0x06756] = { [0x0FE00] = 0x0FA94 },
        [0x0675E] = { [0x0FE00] = 0x2F8DB },
        [0x0677B] = { [0x0FE00] = 0x0F9C8 },
        [0x06785] = { [0x0FE00] = 0x2F8E0 },
        [0x06797] = { [0x0FE00] = 0x0F9F4 },
        [0x067F3] = { [0x0FE00] = 0x0F9C9 },
        [0x067FA] = { [0x0FE00] = 0x2F8DF },
        [0x06817] = { [0x0FE00] = 0x0F9DA },
        [0x0681F] = { [0x0FE00] = 0x2F8E5 },
        [0x06852] = { [0x0FE00] = 0x2F8E1 },
        [0x06881] = { [0x0FE00] = 0x0F97A },
        [0x06885] = { [0x0FE00] = 0x0FA44, [0x0FE01] = 0x2F8E2 },
        [0x0688E] = { [0x0FE00] = 0x2F8E4 },
        [0x068A8] = { [0x0FE00] = 0x0F9E2 },
        [0x06914] = { [0x0FE00] = 0x2F8E6 },
        [0x06942] = { [0x0FE00] = 0x2F8E8 },
        [0x069A3] = { [0x0FE00] = 0x2F8E9 },
        [0x069EA] = { [0x0FE00] = 0x2F8EA },
        [0x06A02] = { [0x0FE00] = 0x0F914, [0x0FE01] = 0x0F95C, [0x0FE02] = 0x0F9BF },
        [0x06A13] = { [0x0FE00] = 0x0F94C },
        [0x06AA8] = { [0x0FE00] = 0x2F8EB },
        [0x06AD3] = { [0x0FE00] = 0x0F931 },
        [0x06ADB] = { [0x0FE00] = 0x2F8ED },
        [0x06B04] = { [0x0FE00] = 0x0F91D },
        [0x06B21] = { [0x0FE00] = 0x2F8EF },
        [0x06B54] = { [0x0FE00] = 0x2F8F1 },
        [0x06B72] = { [0x0FE00] = 0x2F8F3 },
        [0x06B77] = { [0x0FE00] = 0x0F98C },
        [0x06B79] = { [0x0FE00] = 0x0FA95 },
        [0x06B9F] = { [0x0FE00] = 0x2F8F4 },
        [0x06BAE] = { [0x0FE00] = 0x0F9A5 },
        [0x06BBA] = { [0x0FE00] = 0x0F970, [0x0FE01] = 0x0FA96, [0x0FE02] = 0x2F8F5 },
        [0x06BBB] = { [0x0FE00] = 0x2F8F6 },
        [0x06C4E] = { [0x0FE00] = 0x2F8FA },
        [0x06C67] = { [0x0FE00] = 0x2F8FE },
        [0x06C88] = { [0x0FE00] = 0x0F972 },
        [0x06CBF] = { [0x0FE00] = 0x2F8FC },
        [0x06CCC] = { [0x0FE00] = 0x0F968 },
        [0x06CCD] = { [0x0FE00] = 0x2F8FD },
        [0x06CE5] = { [0x0FE00] = 0x0F9E3 },
        [0x06D16] = { [0x0FE00] = 0x2F8FF },
        [0x06D1B] = { [0x0FE00] = 0x0F915 },
        [0x06D1E] = { [0x0FE00] = 0x0FA05 },
        [0x06D34] = { [0x0FE00] = 0x2F907 },
        [0x06D3E] = { [0x0FE00] = 0x2F900 },
        [0x06D41] = { [0x0FE00] = 0x0F9CA, [0x0FE01] = 0x0FA97, [0x0FE02] = 0x2F902 },
        [0x06D69] = { [0x0FE00] = 0x2F903 },
        [0x06D6A] = { [0x0FE00] = 0x0F92A },
        [0x06D77] = { [0x0FE00] = 0x0FA45, [0x0FE01] = 0x2F901 },
        [0x06D78] = { [0x0FE00] = 0x2F904 },
        [0x06D85] = { [0x0FE00] = 0x2F905 },
        [0x06DCB] = { [0x0FE00] = 0x0F9F5 },
        [0x06DDA] = { [0x0FE00] = 0x0F94D },
        [0x06DEA] = { [0x0FE00] = 0x0F9D6 },
        [0x06DF9] = { [0x0FE00] = 0x2F90E },
        [0x06E1A] = { [0x0FE00] = 0x0FA46 },
        [0x06E2F] = { [0x0FE00] = 0x2F908 },
        [0x06E6E] = { [0x0FE00] = 0x2F909 },
        [0x06E9C] = { [0x0FE00] = 0x0F9CB },
        [0x06EBA] = { [0x0FE00] = 0x0F9EC },
        [0x06EC7] = { [0x0FE00] = 0x2F90C },
        [0x06ECB] = { [0x0FE00] = 0x0FA99, [0x0FE01] = 0x2F90B },
        [0x06ED1] = { [0x0FE00] = 0x0F904 },
        [0x06EDB] = { [0x0FE00] = 0x0FA98 },
        [0x06F0F] = { [0x0FE00] = 0x0F94E },
        [0x06F22] = { [0x0FE00] = 0x0FA47, [0x0FE01] = 0x0FA9A },
        [0x06F23] = { [0x0FE00] = 0x0F992 },
        [0x06F6E] = { [0x0FE00] = 0x2F90F },
        [0x06FC6] = { [0x0FE00] = 0x2F912 },
        [0x06FEB] = { [0x0FE00] = 0x0F922 },
        [0x06FFE] = { [0x0FE00] = 0x0F984 },
        [0x0701B] = { [0x0FE00] = 0x2F915 },
        [0x0701E] = { [0x0FE00] = 0x0FA9B, [0x0FE01] = 0x2F914 },
        [0x07039] = { [0x0FE00] = 0x2F913 },
        [0x0704A] = { [0x0FE00] = 0x2F917 },
        [0x07070] = { [0x0FE00] = 0x2F835 },
        [0x07077] = { [0x0FE00] = 0x2F919 },
        [0x0707D] = { [0x0FE00] = 0x2F918 },
        [0x07099] = { [0x0FE00] = 0x0F9FB },
        [0x070AD] = { [0x0FE00] = 0x2F91A },
        [0x070C8] = { [0x0FE00] = 0x0F99F },
        [0x070D9] = { [0x0FE00] = 0x0F916 },
        [0x07145] = { [0x0FE00] = 0x2F91C },
        [0x07149] = { [0x0FE00] = 0x0F993 },
        [0x0716E] = { [0x0FE00] = 0x0FA48, [0x0FE01] = 0x0FA9C },
        [0x0719C] = { [0x0FE00] = 0x2F91E },
        [0x071CE] = { [0x0FE00] = 0x0F9C0 },
        [0x071D0] = { [0x0FE00] = 0x0F9EE },
        [0x07210] = { [0x0FE00] = 0x0F932 },
        [0x0721B] = { [0x0FE00] = 0x0F91E },
        [0x07228] = { [0x0FE00] = 0x2F920 },
        [0x0722B] = { [0x0FE00] = 0x0FA49 },
        [0x07235] = { [0x0FE00] = 0x0FA9E, [0x0FE01] = 0x2F921 },
        [0x07250] = { [0x0FE00] = 0x2F922 },
        [0x07262] = { [0x0FE00] = 0x0F946 },
        [0x07280] = { [0x0FE00] = 0x2F924 },
        [0x07295] = { [0x0FE00] = 0x2F925 },
        [0x072AF] = { [0x0FE00] = 0x0FA9F },
        [0x072C0] = { [0x0FE00] = 0x0F9FA },
        [0x072FC] = { [0x0FE00] = 0x0F92B },
        [0x0732A] = { [0x0FE00] = 0x0FA16, [0x0FE01] = 0x0FAA0 },
        [0x07375] = { [0x0FE00] = 0x0F9A7 },
        [0x0737A] = { [0x0FE00] = 0x2F928 },
        [0x07387] = { [0x0FE00] = 0x0F961, [0x0FE01] = 0x0F9DB },
        [0x0738B] = { [0x0FE00] = 0x2F929 },
        [0x073A5] = { [0x0FE00] = 0x2F92B },
        [0x073B2] = { [0x0FE00] = 0x0F9AD },
        [0x073DE] = { [0x0FE00] = 0x0F917 },
        [0x07406] = { [0x0FE00] = 0x0F9E4 },
        [0x07409] = { [0x0FE00] = 0x0F9CC },
        [0x07422] = { [0x0FE00] = 0x0FA4A },
        [0x07447] = { [0x0FE00] = 0x2F92E },
        [0x0745C] = { [0x0FE00] = 0x2F92F },
        [0x07469] = { [0x0FE00] = 0x0F9AE },
        [0x07471] = { [0x0FE00] = 0x0FAA1, [0x0FE01] = 0x2F930 },
        [0x07485] = { [0x0FE00] = 0x2F931 },
        [0x07489] = { [0x0FE00] = 0x0F994 },
        [0x07498] = { [0x0FE00] = 0x0F9EF },
        [0x074CA] = { [0x0FE00] = 0x2F932 },
        [0x07506] = { [0x0FE00] = 0x0FAA2 },
        [0x07524] = { [0x0FE00] = 0x2F934 },
        [0x0753B] = { [0x0FE00] = 0x0FAA3 },
        [0x0753E] = { [0x0FE00] = 0x2F936 },
        [0x07559] = { [0x0FE00] = 0x0F9CD },
        [0x07565] = { [0x0FE00] = 0x0F976 },
        [0x07570] = { [0x0FE00] = 0x0F962, [0x0FE01] = 0x2F938 },
        [0x075E2] = { [0x0FE00] = 0x0F9E5 },
        [0x07610] = { [0x0FE00] = 0x2F93A },
        [0x0761D] = { [0x0FE00] = 0x0FAA4 },
        [0x0761F] = { [0x0FE00] = 0x0FAA5 },
        [0x07642] = { [0x0FE00] = 0x0F9C1 },
        [0x07669] = { [0x0FE00] = 0x0F90E },
        [0x076CA] = { [0x0FE00] = 0x0FA17, [0x0FE01] = 0x0FAA6 },
        [0x076DB] = { [0x0FE00] = 0x0FAA7 },
        [0x076E7] = { [0x0FE00] = 0x0F933 },
        [0x076F4] = { [0x0FE00] = 0x0FAA8, [0x0FE01] = 0x2F940 },
        [0x07701] = { [0x0FE00] = 0x0F96D },
        [0x0771E] = { [0x0FE00] = 0x2F945 },
        [0x0771F] = { [0x0FE00] = 0x2F946, [0x0FE01] = 0x2F947 },
        [0x07740] = { [0x0FE00] = 0x0FAAA },
        [0x0774A] = { [0x0FE00] = 0x0FAA9, [0x0FE01] = 0x2F948 },
        [0x0778B] = { [0x0FE00] = 0x2F94A },
        [0x077A7] = { [0x0FE00] = 0x0FA9D },
        [0x0784E] = { [0x0FE00] = 0x2F94E },
        [0x0786B] = { [0x0FE00] = 0x0F9CE },
        [0x0788C] = { [0x0FE00] = 0x0F93B, [0x0FE01] = 0x2F94F },
        [0x07891] = { [0x0FE00] = 0x0FA4B },
        [0x078CA] = { [0x0FE00] = 0x0F947 },
        [0x078CC] = { [0x0FE00] = 0x0FAAB, [0x0FE01] = 0x2F950 },
        [0x078FB] = { [0x0FE00] = 0x0F964 },
        [0x0792A] = { [0x0FE00] = 0x0F985 },
        [0x0793C] = { [0x0FE00] = 0x0FA18 },
        [0x0793E] = { [0x0FE00] = 0x0FA4C },
        [0x07948] = { [0x0FE00] = 0x0FA4E },
        [0x07949] = { [0x0FE00] = 0x0FA4D },
        [0x07950] = { [0x0FE00] = 0x0FA4F },
        [0x07956] = { [0x0FE00] = 0x0FA50, [0x0FE01] = 0x2F953 },
        [0x0795D] = { [0x0FE00] = 0x0FA51 },
        [0x0795E] = { [0x0FE00] = 0x0FA19 },
        [0x07965] = { [0x0FE00] = 0x0FA1A },
        [0x0797F] = { [0x0FE00] = 0x0F93C },
        [0x0798D] = { [0x0FE00] = 0x0FA52 },
        [0x0798E] = { [0x0FE00] = 0x0FA53 },
        [0x0798F] = { [0x0FE00] = 0x0FA1B, [0x0FE01] = 0x2F956 },
        [0x079AE] = { [0x0FE00] = 0x0F9B6 },
        [0x079CA] = { [0x0FE00] = 0x0F995 },
        [0x079EB] = { [0x0FE00] = 0x2F957 },
        [0x07A1C] = { [0x0FE00] = 0x0F956 },
        [0x07A40] = { [0x0FE00] = 0x0FA54, [0x0FE01] = 0x2F959 },
        [0x07A4A] = { [0x0FE00] = 0x2F95A },
        [0x07A4F] = { [0x0FE00] = 0x2F95B },
        [0x07A81] = { [0x0FE00] = 0x0FA55 },
        [0x07AB1] = { [0x0FE00] = 0x0FAAC },
        [0x07ACB] = { [0x0FE00] = 0x0F9F7 },
        [0x07AEE] = { [0x0FE00] = 0x2F95F },
        [0x07B20] = { [0x0FE00] = 0x0F9F8 },
        [0x07BC0] = { [0x0FE00] = 0x0FA56, [0x0FE01] = 0x0FAAD },
        [0x07BC6] = { [0x0FE00] = 0x2F962 },
        [0x07BC9] = { [0x0FE00] = 0x2F963 },
        [0x07C3E] = { [0x0FE00] = 0x0F9A6 },
        [0x07C60] = { [0x0FE00] = 0x0F944 },
        [0x07C7B] = { [0x0FE00] = 0x0FAAE },
        [0x07C92] = { [0x0FE00] = 0x0F9F9 },
        [0x07CBE] = { [0x0FE00] = 0x0FA1D },
        [0x07CD2] = { [0x0FE00] = 0x2F966 },
        [0x07CD6] = { [0x0FE00] = 0x0FA03 },
        [0x07CE3] = { [0x0FE00] = 0x2F969 },
        [0x07CE7] = { [0x0FE00] = 0x0F97B },
        [0x07CE8] = { [0x0FE00] = 0x2F968 },
        [0x07D00] = { [0x0FE00] = 0x2F96A },
        [0x07D10] = { [0x0FE00] = 0x0F9CF },
        [0x07D22] = { [0x0FE00] = 0x0F96A },
        [0x07D2F] = { [0x0FE00] = 0x0F94F },
        [0x07D5B] = { [0x0FE00] = 0x0FAAF },
        [0x07D63] = { [0x0FE00] = 0x2F96C },
        [0x07DA0] = { [0x0FE00] = 0x0F93D },
        [0x07DBE] = { [0x0FE00] = 0x0F957 },
        [0x07DC7] = { [0x0FE00] = 0x2F96E },
        [0x07DF4] = { [0x0FE00] = 0x0F996, [0x0FE01] = 0x0FA57, [0x0FE02] = 0x0FAB0 },
        [0x07E02] = { [0x0FE00] = 0x2F96F },
        [0x07E09] = { [0x0FE00] = 0x0FA58 },
        [0x07E37] = { [0x0FE00] = 0x0F950 },
        [0x07E41] = { [0x0FE00] = 0x0FA59 },
        [0x07E45] = { [0x0FE00] = 0x2F970 },
        [0x07F3E] = { [0x0FE00] = 0x0FAB1 },
        [0x07F72] = { [0x0FE00] = 0x0FA5A },
        [0x07F79] = { [0x0FE00] = 0x0F9E6 },
        [0x07F7A] = { [0x0FE00] = 0x2F976 },
        [0x07F85] = { [0x0FE00] = 0x0F90F },
        [0x07F95] = { [0x0FE00] = 0x2F978 },
        [0x07F9A] = { [0x0FE00] = 0x0F9AF },
        [0x07FBD] = { [0x0FE00] = 0x0FA1E },
        [0x07FFA] = { [0x0FE00] = 0x2F979 },
        [0x08001] = { [0x0FE00] = 0x0F934 },
        [0x08005] = { [0x0FE00] = 0x0FA5B, [0x0FE01] = 0x0FAB2, [0x0FE02] = 0x2F97A },
        [0x08046] = { [0x0FE00] = 0x0F9B0 },
        [0x08060] = { [0x0FE00] = 0x2F97D },
        [0x0806F] = { [0x0FE00] = 0x0F997 },
        [0x08070] = { [0x0FE00] = 0x2F97F },
        [0x0807E] = { [0x0FE00] = 0x0F945 },
        [0x0808B] = { [0x0FE00] = 0x0F953 },
        [0x080AD] = { [0x0FE00] = 0x2F8D6 },
        [0x080B2] = { [0x0FE00] = 0x2F982 },
        [0x08103] = { [0x0FE00] = 0x2F983 },
        [0x0813E] = { [0x0FE00] = 0x2F985 },
        [0x081D8] = { [0x0FE00] = 0x0F926 },
        [0x081E8] = { [0x0FE00] = 0x0F9F6 },
        [0x081ED] = { [0x0FE00] = 0x0FA5C },
        [0x08201] = { [0x0FE00] = 0x2F893, [0x0FE01] = 0x2F98B },
        [0x08204] = { [0x0FE00] = 0x2F98C },
        [0x08218] = { [0x0FE00] = 0x0FA6D },
        [0x0826F] = { [0x0FE00] = 0x0F97C },
        [0x08279] = { [0x0FE00] = 0x0FA5D, [0x0FE01] = 0x0FA5E },
        [0x0828B] = { [0x0FE00] = 0x2F990 },
        [0x08291] = { [0x0FE00] = 0x2F98F },
        [0x0829D] = { [0x0FE00] = 0x2F991 },
        [0x082B1] = { [0x0FE00] = 0x2F993 },
        [0x082B3] = { [0x0FE00] = 0x2F994 },
        [0x082BD] = { [0x0FE00] = 0x2F995 },
        [0x082E5] = { [0x0FE00] = 0x0F974, [0x0FE01] = 0x2F998 },
        [0x082E6] = { [0x0FE00] = 0x2F996 },
        [0x0831D] = { [0x0FE00] = 0x2F999 },
        [0x08323] = { [0x0FE00] = 0x2F99C },
        [0x08336] = { [0x0FE00] = 0x0F9FE },
        [0x08352] = { [0x0FE00] = 0x0FAB3 },
        [0x08353] = { [0x0FE00] = 0x2F9A0 },
        [0x08363] = { [0x0FE00] = 0x2F99A },
        [0x083AD] = { [0x0FE00] = 0x2F99B },
        [0x083BD] = { [0x0FE00] = 0x2F99D },
        [0x083C9] = { [0x0FE00] = 0x0F93E },
        [0x083CA] = { [0x0FE00] = 0x2F9A1 },
        [0x083CC] = { [0x0FE00] = 0x2F9A2 },
        [0x083DC] = { [0x0FE00] = 0x2F9A3 },
        [0x083E7] = { [0x0FE00] = 0x2F99E },
        [0x083EF] = { [0x0FE00] = 0x0FAB4 },
        [0x083F1] = { [0x0FE00] = 0x0F958 },
        [0x0843D] = { [0x0FE00] = 0x0F918 },
        [0x08449] = { [0x0FE00] = 0x0F96E },
        [0x08457] = { [0x0FE00] = 0x0FA5F, [0x0FE01] = 0x2F99F },
        [0x084EE] = { [0x0FE00] = 0x0F999 },
        [0x084F1] = { [0x0FE00] = 0x2F9A8 },
        [0x084F3] = { [0x0FE00] = 0x2F9A9 },
        [0x084FC] = { [0x0FE00] = 0x0F9C2 },
        [0x08516] = { [0x0FE00] = 0x2F9AA },
        [0x08564] = { [0x0FE00] = 0x2F9AC },
        [0x085CD] = { [0x0FE00] = 0x0F923 },
        [0x085FA] = { [0x0FE00] = 0x0F9F0 },
        [0x08606] = { [0x0FE00] = 0x0F935 },
        [0x08612] = { [0x0FE00] = 0x0FA20 },
        [0x0862D] = { [0x0FE00] = 0x0F91F },
        [0x0863F] = { [0x0FE00] = 0x0F910 },
        [0x08650] = { [0x0FE00] = 0x2F9B3 },
        [0x0865C] = { [0x0FE00] = 0x0F936, [0x0FE01] = 0x2F9B4 },
        [0x08667] = { [0x0FE00] = 0x2F9B5 },
        [0x08669] = { [0x0FE00] = 0x2F9B6 },
        [0x08688] = { [0x0FE00] = 0x2F9B8 },
        [0x086A9] = { [0x0FE00] = 0x2F9B7 },
        [0x086E2] = { [0x0FE00] = 0x2F9BA },
        [0x0870E] = { [0x0FE00] = 0x2F9B9 },
        [0x08728] = { [0x0FE00] = 0x2F9BC },
        [0x0876B] = { [0x0FE00] = 0x2F9BD },
        [0x08779] = { [0x0FE00] = 0x0FAB5, [0x0FE01] = 0x2F9BB },
        [0x08786] = { [0x0FE00] = 0x2F9BE },
        [0x087BA] = { [0x0FE00] = 0x0F911 },
        [0x087E1] = { [0x0FE00] = 0x2F9C0 },
        [0x08801] = { [0x0FE00] = 0x2F9C1 },
        [0x0881F] = { [0x0FE00] = 0x0F927 },
        [0x0884C] = { [0x0FE00] = 0x0FA08 },
        [0x08860] = { [0x0FE00] = 0x2F9C3 },
        [0x08863] = { [0x0FE00] = 0x2F9C4 },
        [0x088C2] = { [0x0FE00] = 0x0F9A0 },
        [0x088CF] = { [0x0FE00] = 0x0F9E7 },
        [0x088D7] = { [0x0FE00] = 0x2F9C6 },
        [0x088DE] = { [0x0FE00] = 0x2F9C7 },
        [0x088E1] = { [0x0FE00] = 0x0F9E8 },
        [0x088F8] = { [0x0FE00] = 0x0F912 },
        [0x088FA] = { [0x0FE00] = 0x2F9C9 },
        [0x08910] = { [0x0FE00] = 0x0FA60 },
        [0x08941] = { [0x0FE00] = 0x0FAB6 },
        [0x08964] = { [0x0FE00] = 0x0F924 },
        [0x08986] = { [0x0FE00] = 0x0FAB7 },
        [0x0898B] = { [0x0FE00] = 0x0FA0A },
        [0x08996] = { [0x0FE00] = 0x0FA61, [0x0FE01] = 0x0FAB8 },
        [0x08AA0] = { [0x0FE00] = 0x2F9CF },
        [0x08AAA] = { [0x0FE00] = 0x0F96F, [0x0FE01] = 0x0F9A1 },
        [0x08ABF] = { [0x0FE00] = 0x0FAB9 },
        [0x08ACB] = { [0x0FE00] = 0x0FABB },
        [0x08AD2] = { [0x0FE00] = 0x0F97D },
        [0x08AD6] = { [0x0FE00] = 0x0F941 },
        [0x08AED] = { [0x0FE00] = 0x0FABE, [0x0FE01] = 0x2F9D0 },
        [0x08AF8] = { [0x0FE00] = 0x0FA22, [0x0FE01] = 0x0FABA },
        [0x08AFE] = { [0x0FE00] = 0x0F95D, [0x0FE01] = 0x0FABD },
        [0x08B01] = { [0x0FE00] = 0x0FA62, [0x0FE01] = 0x0FABC },
        [0x08B39] = { [0x0FE00] = 0x0FA63, [0x0FE01] = 0x0FABF },
        [0x08B58] = { [0x0FE00] = 0x0F9FC },
        [0x08B80] = { [0x0FE00] = 0x0F95A },
        [0x08B8A] = { [0x0FE00] = 0x0FAC0, [0x0FE01] = 0x2F9D1 },
        [0x08C48] = { [0x0FE00] = 0x0F900 },
        [0x08C55] = { [0x0FE00] = 0x2F9D2 },
        [0x08CAB] = { [0x0FE00] = 0x2F9D4 },
        [0x08CC1] = { [0x0FE00] = 0x2F9D5 },
        [0x08CC2] = { [0x0FE00] = 0x0F948 },
        [0x08CC8] = { [0x0FE00] = 0x0F903 },
        [0x08CD3] = { [0x0FE00] = 0x0FA64 },
        [0x08D08] = { [0x0FE00] = 0x0FA65, [0x0FE01] = 0x0FAC1 },
        [0x08D1B] = { [0x0FE00] = 0x2F9D6 },
        [0x08D77] = { [0x0FE00] = 0x2F9D7 },
        [0x08DBC] = { [0x0FE00] = 0x2F9DB },
        [0x08DCB] = { [0x0FE00] = 0x2F9DA },
        [0x08DEF] = { [0x0FE00] = 0x0F937 },
        [0x08DF0] = { [0x0FE00] = 0x2F9DC },
        [0x08ECA] = { [0x0FE00] = 0x0F902 },
        [0x08ED4] = { [0x0FE00] = 0x2F9DE },
        [0x08F26] = { [0x0FE00] = 0x0F998 },
        [0x08F2A] = { [0x0FE00] = 0x0F9D7 },
        [0x08F38] = { [0x0FE00] = 0x0FAC2, [0x0FE01] = 0x2F9DF },
        [0x08F3B] = { [0x0FE00] = 0x0FA07 },
        [0x08F62] = { [0x0FE00] = 0x0F98D },
        [0x08F9E] = { [0x0FE00] = 0x2F98D },
        [0x08FB0] = { [0x0FE00] = 0x0F971 },
        [0x08FB6] = { [0x0FE00] = 0x0FA66 },
        [0x09023] = { [0x0FE00] = 0x0F99A },
        [0x09038] = { [0x0FE00] = 0x0FA25, [0x0FE01] = 0x0FA67 },
        [0x09072] = { [0x0FE00] = 0x0FAC3 },
        [0x0907C] = { [0x0FE00] = 0x0F9C3 },
        [0x0908F] = { [0x0FE00] = 0x0F913 },
        [0x09094] = { [0x0FE00] = 0x2F9E2 },
        [0x090CE] = { [0x0FE00] = 0x0F92C },
        [0x090DE] = { [0x0FE00] = 0x0FA2E },
        [0x090F1] = { [0x0FE00] = 0x2F9E3 },
        [0x090FD] = { [0x0FE00] = 0x0FA26 },
        [0x09111] = { [0x0FE00] = 0x2F9E4 },
        [0x0911B] = { [0x0FE00] = 0x2F9E6 },
        [0x0916A] = { [0x0FE00] = 0x0F919 },
        [0x09199] = { [0x0FE00] = 0x0FAC4 },
        [0x091B4] = { [0x0FE00] = 0x0F9B7 },
        [0x091CC] = { [0x0FE00] = 0x0F9E9 },
        [0x091CF] = { [0x0FE00] = 0x0F97E },
        [0x091D1] = { [0x0FE00] = 0x0F90A },
        [0x09234] = { [0x0FE00] = 0x0F9B1 },
        [0x09238] = { [0x0FE00] = 0x2F9E7 },
        [0x09276] = { [0x0FE00] = 0x0FAC5 },
        [0x0927C] = { [0x0FE00] = 0x2F9EA },
        [0x092D7] = { [0x0FE00] = 0x2F9E8 },
        [0x092D8] = { [0x0FE00] = 0x2F9E9 },
        [0x09304] = { [0x0FE00] = 0x0F93F },
        [0x0934A] = { [0x0FE00] = 0x0F99B },
        [0x093F9] = { [0x0FE00] = 0x2F9EB },
        [0x09415] = { [0x0FE00] = 0x2F9EC },
        [0x0958B] = { [0x0FE00] = 0x2F9EE },
        [0x095AD] = { [0x0FE00] = 0x0F986 },
        [0x095B7] = { [0x0FE00] = 0x2F9F0 },
        [0x0962E] = { [0x0FE00] = 0x0F9C6 },
        [0x0964B] = { [0x0FE00] = 0x0F951 },
        [0x0964D] = { [0x0FE00] = 0x0FA09 },
        [0x09675] = { [0x0FE00] = 0x0F959 },
        [0x09678] = { [0x0FE00] = 0x0F9D3 },
        [0x0967C] = { [0x0FE00] = 0x0FAC6 },
        [0x09686] = { [0x0FE00] = 0x0F9DC },
        [0x096A3] = { [0x0FE00] = 0x0F9F1 },
        [0x096B7] = { [0x0FE00] = 0x0FA2F },
        [0x096B8] = { [0x0FE00] = 0x0F9B8 },
        [0x096C3] = { [0x0FE00] = 0x2F9F3 },
        [0x096E2] = { [0x0FE00] = 0x0F9EA },
        [0x096E3] = { [0x0FE00] = 0x0FA68, [0x0FE01] = 0x0FAC7 },
        [0x096F6] = { [0x0FE00] = 0x0F9B2 },
        [0x096F7] = { [0x0FE00] = 0x0F949 },
        [0x09723] = { [0x0FE00] = 0x2F9F5 },
        [0x09732] = { [0x0FE00] = 0x0F938 },
        [0x09748] = { [0x0FE00] = 0x0F9B3 },
        [0x09756] = { [0x0FE00] = 0x0FA1C, [0x0FE01] = 0x0FAC8 },
        [0x097DB] = { [0x0FE00] = 0x0FAC9 },
        [0x097E0] = { [0x0FE00] = 0x2F9FA },
        [0x097FF] = { [0x0FE00] = 0x0FA69, [0x0FE01] = 0x0FACA },
        [0x0980B] = { [0x0FE00] = 0x0FACB, [0x0FE01] = 0x2F9FE, [0x0FE02] = 0x2F9FF },
        [0x09818] = { [0x0FE00] = 0x0F9B4 },
        [0x09829] = { [0x0FE00] = 0x2FA00 },
        [0x0983B] = { [0x0FE00] = 0x0FA6A, [0x0FE01] = 0x0FACC },
        [0x0985E] = { [0x0FE00] = 0x0F9D0 },
        [0x098E2] = { [0x0FE00] = 0x2FA02 },
        [0x098EF] = { [0x0FE00] = 0x0FA2A },
        [0x098FC] = { [0x0FE00] = 0x0FA2B },
        [0x09928] = { [0x0FE00] = 0x0FA2C },
        [0x09929] = { [0x0FE00] = 0x2FA04 },
        [0x099A7] = { [0x0FE00] = 0x2FA05 },
        [0x099C2] = { [0x0FE00] = 0x2FA06 },
        [0x099F1] = { [0x0FE00] = 0x0F91A },
        [0x099FE] = { [0x0FE00] = 0x2FA07 },
        [0x09A6A] = { [0x0FE00] = 0x0F987 },
        [0x09B12] = { [0x0FE00] = 0x0FACD, [0x0FE01] = 0x2FA0A },
        [0x09B6F] = { [0x0FE00] = 0x0F939 },
        [0x09C40] = { [0x0FE00] = 0x2FA0B },
        [0x09C57] = { [0x0FE00] = 0x0F9F2 },
        [0x09CFD] = { [0x0FE00] = 0x2FA0C },
        [0x09D67] = { [0x0FE00] = 0x2FA0F },
        [0x09DB4] = { [0x0FE00] = 0x0FA2D },
        [0x09DFA] = { [0x0FE00] = 0x0F93A },
        [0x09E1E] = { [0x0FE00] = 0x0F920 },
        [0x09E7F] = { [0x0FE00] = 0x0F940 },
        [0x09E97] = { [0x0FE00] = 0x0F988 },
        [0x09E9F] = { [0x0FE00] = 0x0F9F3 },
        [0x09EBB] = { [0x0FE00] = 0x2FA15 },
        [0x09ECE] = { [0x0FE00] = 0x0F989 },
        [0x09EF9] = { [0x0FE00] = 0x2FA17 },
        [0x09EFE] = { [0x0FE00] = 0x2FA18 },
        [0x09F05] = { [0x0FE00] = 0x2FA19 },
        [0x09F0F] = { [0x0FE00] = 0x2FA1A },
        [0x09F16] = { [0x0FE00] = 0x2FA1B },
        [0x09F3B] = { [0x0FE00] = 0x2FA1C },
        [0x09F43] = { [0x0FE00] = 0x0FAD8 },
        [0x09F8D] = { [0x0FE00] = 0x0F9C4 },
        [0x09F8E] = { [0x0FE00] = 0x0FAD9 },
        [0x09F9C] = { [0x0FE00] = 0x0F907, [0x0FE01] = 0x0F908, [0x0FE02] = 0x0FACE },
        [0x20122] = { [0x0FE00] = 0x2F803 },
        [0x2051C] = { [0x0FE00] = 0x2F812 },
        [0x20525] = { [0x0FE00] = 0x2F91B },
        [0x2054B] = { [0x0FE00] = 0x2F816 },
        [0x2063A] = { [0x0FE00] = 0x2F80D },
        [0x20804] = { [0x0FE00] = 0x2F9D9 },
        [0x208DE] = { [0x0FE00] = 0x2F9DD },
        [0x20A2C] = { [0x0FE00] = 0x2F834 },
        [0x20B63] = { [0x0FE00] = 0x2F838 },
        [0x214E4] = { [0x0FE00] = 0x2F859 },
        [0x216A8] = { [0x0FE00] = 0x2F860 },
        [0x216EA] = { [0x0FE00] = 0x2F861 },
        [0x219C8] = { [0x0FE00] = 0x2F86C },
        [0x21B18] = { [0x0FE00] = 0x2F871 },
        [0x21D0B] = { [0x0FE00] = 0x2F8F8 },
        [0x21DE4] = { [0x0FE00] = 0x2F87B },
        [0x21DE6] = { [0x0FE00] = 0x2F87D },
        [0x22183] = { [0x0FE00] = 0x2F889 },
        [0x2219F] = { [0x0FE00] = 0x2F939 },
        [0x22331] = { [0x0FE00] = 0x2F891, [0x0FE01] = 0x2F892 },
        [0x226D4] = { [0x0FE00] = 0x2F8A4 },
        [0x22844] = { [0x0FE00] = 0x0FAD0 },
        [0x2284A] = { [0x0FE00] = 0x0FACF },
        [0x22B0C] = { [0x0FE00] = 0x2F8B8 },
        [0x22BF1] = { [0x0FE00] = 0x2F8BE },
        [0x2300A] = { [0x0FE00] = 0x2F8CA },
        [0x232B8] = { [0x0FE00] = 0x2F897 },
        [0x2335F] = { [0x0FE00] = 0x2F980 },
        [0x23393] = { [0x0FE00] = 0x2F989 },
        [0x2339C] = { [0x0FE00] = 0x2F98A },
        [0x233C3] = { [0x0FE00] = 0x2F8DD },
        [0x233D5] = { [0x0FE00] = 0x0FAD1 },
        [0x2346D] = { [0x0FE00] = 0x2F8E3 },
        [0x236A3] = { [0x0FE00] = 0x2F8EC },
        [0x238A7] = { [0x0FE00] = 0x2F8F0 },
        [0x23A8D] = { [0x0FE00] = 0x2F8F7 },
        [0x23AFA] = { [0x0FE00] = 0x2F8F9 },
        [0x23CBC] = { [0x0FE00] = 0x2F8FB },
        [0x23D1E] = { [0x0FE00] = 0x2F906 },
        [0x23ED1] = { [0x0FE00] = 0x2F90D },
        [0x23F5E] = { [0x0FE00] = 0x2F910 },
        [0x23F8E] = { [0x0FE00] = 0x2F911 },
        [0x24263] = { [0x0FE00] = 0x2F91D },
        [0x242EE] = { [0x0FE00] = 0x0FA6C },
        [0x243AB] = { [0x0FE00] = 0x2F91F },
        [0x24608] = { [0x0FE00] = 0x2F923 },
        [0x24735] = { [0x0FE00] = 0x2F926 },
        [0x24814] = { [0x0FE00] = 0x2F927 },
        [0x24C36] = { [0x0FE00] = 0x2F935 },
        [0x24C92] = { [0x0FE00] = 0x2F937 },
        [0x24FA1] = { [0x0FE00] = 0x2F93B },
        [0x24FB8] = { [0x0FE00] = 0x2F93C },
        [0x25044] = { [0x0FE00] = 0x2F93D },
        [0x250F2] = { [0x0FE00] = 0x2F942 },
        [0x250F3] = { [0x0FE00] = 0x2F941 },
        [0x25119] = { [0x0FE00] = 0x2F943 },
        [0x25133] = { [0x0FE00] = 0x2F944 },
        [0x25249] = { [0x0FE00] = 0x0FAD5 },
        [0x2541D] = { [0x0FE00] = 0x2F94D },
        [0x25626] = { [0x0FE00] = 0x2F952 },
        [0x2569A] = { [0x0FE00] = 0x2F954 },
        [0x256C5] = { [0x0FE00] = 0x2F955 },
        [0x2597C] = { [0x0FE00] = 0x2F95C },
        [0x25AA7] = { [0x0FE00] = 0x2F95D, [0x0FE01] = 0x2F95E },
        [0x25BAB] = { [0x0FE00] = 0x2F961 },
        [0x25C80] = { [0x0FE00] = 0x2F965 },
        [0x25CD0] = { [0x0FE00] = 0x0FAD6 },
        [0x25F86] = { [0x0FE00] = 0x2F96B },
        [0x261DA] = { [0x0FE00] = 0x2F898 },
        [0x26228] = { [0x0FE00] = 0x2F972 },
        [0x26247] = { [0x0FE00] = 0x2F973 },
        [0x262D9] = { [0x0FE00] = 0x2F975 },
        [0x2633E] = { [0x0FE00] = 0x2F977 },
        [0x264DA] = { [0x0FE00] = 0x2F97B },
        [0x26523] = { [0x0FE00] = 0x2F97C },
        [0x265A8] = { [0x0FE00] = 0x2F97E },
        [0x267A7] = { [0x0FE00] = 0x2F987 },
        [0x267B5] = { [0x0FE00] = 0x2F988 },
        [0x26B3C] = { [0x0FE00] = 0x2F997 },
        [0x26C36] = { [0x0FE00] = 0x2F9A4 },
        [0x26CD5] = { [0x0FE00] = 0x2F9A6 },
        [0x26D6B] = { [0x0FE00] = 0x2F9A5 },
        [0x26F2C] = { [0x0FE00] = 0x2F9AD },
        [0x26FB1] = { [0x0FE00] = 0x2F9B0 },
        [0x270D2] = { [0x0FE00] = 0x2F9B1 },
        [0x273CA] = { [0x0FE00] = 0x2F9AB },
        [0x27667] = { [0x0FE00] = 0x2F9C5 },
        [0x278AE] = { [0x0FE00] = 0x2F9CB },
        [0x27966] = { [0x0FE00] = 0x2F9CC },
        [0x27CA8] = { [0x0FE00] = 0x2F9D3 },
        [0x27ED3] = { [0x0FE00] = 0x0FAD7 },
        [0x27F2F] = { [0x0FE00] = 0x2F9D8 },
        [0x285D2] = { [0x0FE00] = 0x2F9E0 },
        [0x285ED] = { [0x0FE00] = 0x2F9E1 },
        [0x2872E] = { [0x0FE00] = 0x2F9E5 },
        [0x28BFA] = { [0x0FE00] = 0x2F9ED },
        [0x28D77] = { [0x0FE00] = 0x2F9F1 },
        [0x29145] = { [0x0FE00] = 0x2F9F6 },
        [0x291DF] = { [0x0FE00] = 0x2F81C },
        [0x2921A] = { [0x0FE00] = 0x2F9F7 },
        [0x2940A] = { [0x0FE00] = 0x2F9FB },
        [0x29496] = { [0x0FE00] = 0x2F9FD },
        [0x295B6] = { [0x0FE00] = 0x2FA01 },
        [0x29B30] = { [0x0FE00] = 0x2FA09 },
        [0x2A0CE] = { [0x0FE00] = 0x2FA10 },
        [0x2A105] = { [0x0FE00] = 0x2FA12 },
        [0x2A20E] = { [0x0FE00] = 0x2FA13 },
        [0x2A291] = { [0x0FE00] = 0x2FA14 },
        [0x2A392] = { [0x0FE00] = 0x2F88F },
        [0x2A600] = { [0x0FE00] = 0x2FA1D },
    }

    local variants = setmetatableindex(function(t,k)
        local kv = knownvariants[k]
        if kv then
            for k, v in next, kv do
                kv[k] = formatters["CJK COMPATIBILITY IDEOGRAPH-%X"](v)
            end
        end
        knownvariants[k] = nil
        return kv
    end)

    do

        local common = {
            category    = "lo",
            cjkwd       = "w",
            description = "<CJK IDEOGRAPH>",
            direction   = "l",
            linebreak   = "id",
        }

        local metatable = {
            __index = common
        }

        local function extender (k)
            local t = {
             -- shcode      = shcode,
                unicodeslot = k,
                variants    = variants[k],
             -- description = formatters["CJK IDEOGRAPH-%05X"](k)
            }
            setmetatable(t,metatable)
            return t
        end

        ranges[#ranges+1] = {
            name     = "cjk ideograph",
            first    = 0x4E00,
            last     = 0x9FBB,
            common   = common,
            extender = extender,
        }

    end

    -- CJK Ideograph Extension A

    do

        local common = {
            category    = "lo",
            cjkwd       = "w",
            description = "<CJK Ideograph Extension A>",
            direction   = "l",
            linebreak   = "id",
        }

        local metatable = {
            __index = common
        }

        local function extender(k)
            local t = {
                unicodeslot = k,
                variants    = variants[k],
             -- description = formatters["CJK IDEOGRAPH-%05X"](k)
            }
            setmetatable(t,metatable)
            return t
        end

        ranges[#ranges+1] = {
            name     = "cjk ideograph extension a",
            first    = 0x3400,
            last     = 0x4DB5,
            common   = common,
            extender = extender,
        }

    end

    -- CJK Ideograph Extension B

    do

        local common = {
            category    = "lo",
            cjkwd       = "w",
            description = "<CJK Ideograph Extension B>",
            direction   = "l",
            linebreak   = "id",
        }

        local metatable = {
            __index = common
        }

        local function extender(k)
            local t = {
             -- shcode      = shcode,
                unicodeslot = k,
                variants    = variants[k],
             -- description = formatters["CJK IDEOGRAPH-%05X"](k)
            }
            setmetatable(t,metatable)
            return t
        end

        ranges[#ranges+1] = {
            name     = "cjk ideograph extension b",
            first    = 0x20000,
            last     = 0x2A6D6,
            common   = common,
            extender = extender,
        }

    end

    -- there are many more cjk .. todo

end
