﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/iotfleetwise/IoTFleetWise_EXPORTS.h>
#include <aws/iotfleetwise/model/NodeDataType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace IoTFleetWise {
namespace Model {

/**
 * <p>An input component that reports the environmental condition of a vehicle.</p>
 *  <p>You can collect data about fluid levels, temperatures, vibrations, or
 * battery voltage from sensors.</p> <p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/iotfleetwise-2021-06-17/Sensor">AWS
 * API Reference</a></p>
 */
class Sensor {
 public:
  AWS_IOTFLEETWISE_API Sensor() = default;
  AWS_IOTFLEETWISE_API Sensor(Aws::Utils::Json::JsonView jsonValue);
  AWS_IOTFLEETWISE_API Sensor& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_IOTFLEETWISE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The fully qualified name of the sensor. For example, the fully qualified name
   * of a sensor might be <code>Vehicle.Body.Engine.Battery</code>.</p>
   */
  inline const Aws::String& GetFullyQualifiedName() const { return m_fullyQualifiedName; }
  inline bool FullyQualifiedNameHasBeenSet() const { return m_fullyQualifiedNameHasBeenSet; }
  template <typename FullyQualifiedNameT = Aws::String>
  void SetFullyQualifiedName(FullyQualifiedNameT&& value) {
    m_fullyQualifiedNameHasBeenSet = true;
    m_fullyQualifiedName = std::forward<FullyQualifiedNameT>(value);
  }
  template <typename FullyQualifiedNameT = Aws::String>
  Sensor& WithFullyQualifiedName(FullyQualifiedNameT&& value) {
    SetFullyQualifiedName(std::forward<FullyQualifiedNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The specified data type of the sensor. </p>
   */
  inline NodeDataType GetDataType() const { return m_dataType; }
  inline bool DataTypeHasBeenSet() const { return m_dataTypeHasBeenSet; }
  inline void SetDataType(NodeDataType value) {
    m_dataTypeHasBeenSet = true;
    m_dataType = value;
  }
  inline Sensor& WithDataType(NodeDataType value) {
    SetDataType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A brief description of a sensor.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  Sensor& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The scientific unit of measurement for data collected by the sensor.</p>
   */
  inline const Aws::String& GetUnit() const { return m_unit; }
  inline bool UnitHasBeenSet() const { return m_unitHasBeenSet; }
  template <typename UnitT = Aws::String>
  void SetUnit(UnitT&& value) {
    m_unitHasBeenSet = true;
    m_unit = std::forward<UnitT>(value);
  }
  template <typename UnitT = Aws::String>
  Sensor& WithUnit(UnitT&& value) {
    SetUnit(std::forward<UnitT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of possible values a sensor can take.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAllowedValues() const { return m_allowedValues; }
  inline bool AllowedValuesHasBeenSet() const { return m_allowedValuesHasBeenSet; }
  template <typename AllowedValuesT = Aws::Vector<Aws::String>>
  void SetAllowedValues(AllowedValuesT&& value) {
    m_allowedValuesHasBeenSet = true;
    m_allowedValues = std::forward<AllowedValuesT>(value);
  }
  template <typename AllowedValuesT = Aws::Vector<Aws::String>>
  Sensor& WithAllowedValues(AllowedValuesT&& value) {
    SetAllowedValues(std::forward<AllowedValuesT>(value));
    return *this;
  }
  template <typename AllowedValuesT = Aws::String>
  Sensor& AddAllowedValues(AllowedValuesT&& value) {
    m_allowedValuesHasBeenSet = true;
    m_allowedValues.emplace_back(std::forward<AllowedValuesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The specified possible minimum value of the sensor.</p>
   */
  inline double GetMin() const { return m_min; }
  inline bool MinHasBeenSet() const { return m_minHasBeenSet; }
  inline void SetMin(double value) {
    m_minHasBeenSet = true;
    m_min = value;
  }
  inline Sensor& WithMin(double value) {
    SetMin(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The specified possible maximum value of the sensor.</p>
   */
  inline double GetMax() const { return m_max; }
  inline bool MaxHasBeenSet() const { return m_maxHasBeenSet; }
  inline void SetMax(double value) {
    m_maxHasBeenSet = true;
    m_max = value;
  }
  inline Sensor& WithMax(double value) {
    SetMax(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The deprecation message for the node or the branch that was moved or
   * deleted.</p>
   */
  inline const Aws::String& GetDeprecationMessage() const { return m_deprecationMessage; }
  inline bool DeprecationMessageHasBeenSet() const { return m_deprecationMessageHasBeenSet; }
  template <typename DeprecationMessageT = Aws::String>
  void SetDeprecationMessage(DeprecationMessageT&& value) {
    m_deprecationMessageHasBeenSet = true;
    m_deprecationMessage = std::forward<DeprecationMessageT>(value);
  }
  template <typename DeprecationMessageT = Aws::String>
  Sensor& WithDeprecationMessage(DeprecationMessageT&& value) {
    SetDeprecationMessage(std::forward<DeprecationMessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A comment in addition to the description.</p>
   */
  inline const Aws::String& GetComment() const { return m_comment; }
  inline bool CommentHasBeenSet() const { return m_commentHasBeenSet; }
  template <typename CommentT = Aws::String>
  void SetComment(CommentT&& value) {
    m_commentHasBeenSet = true;
    m_comment = std::forward<CommentT>(value);
  }
  template <typename CommentT = Aws::String>
  Sensor& WithComment(CommentT&& value) {
    SetComment(std::forward<CommentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The fully qualified name of the struct node for a sensor if the data type of
   * the actuator is <code>Struct</code> or <code>StructArray</code>. For example,
   * the struct fully qualified name of a sensor might be
   * <code>Vehicle.ADAS.CameraStruct</code>.</p>
   */
  inline const Aws::String& GetStructFullyQualifiedName() const { return m_structFullyQualifiedName; }
  inline bool StructFullyQualifiedNameHasBeenSet() const { return m_structFullyQualifiedNameHasBeenSet; }
  template <typename StructFullyQualifiedNameT = Aws::String>
  void SetStructFullyQualifiedName(StructFullyQualifiedNameT&& value) {
    m_structFullyQualifiedNameHasBeenSet = true;
    m_structFullyQualifiedName = std::forward<StructFullyQualifiedNameT>(value);
  }
  template <typename StructFullyQualifiedNameT = Aws::String>
  Sensor& WithStructFullyQualifiedName(StructFullyQualifiedNameT&& value) {
    SetStructFullyQualifiedName(std::forward<StructFullyQualifiedNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_fullyQualifiedName;
  bool m_fullyQualifiedNameHasBeenSet = false;

  NodeDataType m_dataType{NodeDataType::NOT_SET};
  bool m_dataTypeHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_unit;
  bool m_unitHasBeenSet = false;

  Aws::Vector<Aws::String> m_allowedValues;
  bool m_allowedValuesHasBeenSet = false;

  double m_min{0.0};
  bool m_minHasBeenSet = false;

  double m_max{0.0};
  bool m_maxHasBeenSet = false;

  Aws::String m_deprecationMessage;
  bool m_deprecationMessageHasBeenSet = false;

  Aws::String m_comment;
  bool m_commentHasBeenSet = false;

  Aws::String m_structFullyQualifiedName;
  bool m_structFullyQualifiedNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTFleetWise
}  // namespace Aws
