﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3/S3_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace S3 {
namespace Model {

/**
 * <p>Object Identifier is unique value to identify objects.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/s3-2006-03-01/ObjectIdentifier">AWS
 * API Reference</a></p>
 */
class ObjectIdentifier {
 public:
  AWS_S3_API ObjectIdentifier() = default;
  AWS_S3_API ObjectIdentifier(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_S3_API ObjectIdentifier& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_S3_API void AddToNode(Aws::Utils::Xml::XmlNode& parentNode) const;

  ///@{
  /**
   * <p>Key name of the object.</p>  <p>Replacement must be made for
   * object keys containing special characters (such as carriage returns) when using
   * XML requests. For more information, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-keys.html#object-key-xml-related-constraints">
   * XML related object key constraints</a>.</p>
   */
  inline const Aws::String& GetKey() const { return m_key; }
  inline bool KeyHasBeenSet() const { return m_keyHasBeenSet; }
  template <typename KeyT = Aws::String>
  void SetKey(KeyT&& value) {
    m_keyHasBeenSet = true;
    m_key = std::forward<KeyT>(value);
  }
  template <typename KeyT = Aws::String>
  ObjectIdentifier& WithKey(KeyT&& value) {
    SetKey(std::forward<KeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Version ID for the specific version of the object to delete.</p>
   * <p>This functionality is not supported for directory buckets.</p>
   */
  inline const Aws::String& GetVersionId() const { return m_versionId; }
  inline bool VersionIdHasBeenSet() const { return m_versionIdHasBeenSet; }
  template <typename VersionIdT = Aws::String>
  void SetVersionId(VersionIdT&& value) {
    m_versionIdHasBeenSet = true;
    m_versionId = std::forward<VersionIdT>(value);
  }
  template <typename VersionIdT = Aws::String>
  ObjectIdentifier& WithVersionId(VersionIdT&& value) {
    SetVersionId(std::forward<VersionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An entity tag (ETag) is an identifier assigned by a web server to a specific
   * version of a resource found at a URL. This header field makes the request method
   * conditional on <code>ETags</code>. </p>  <p>Entity tags (ETags) for S3
   * Express One Zone are random alphanumeric strings unique to the object. </p>
   *
   */
  inline const Aws::String& GetETag() const { return m_eTag; }
  inline bool ETagHasBeenSet() const { return m_eTagHasBeenSet; }
  template <typename ETagT = Aws::String>
  void SetETag(ETagT&& value) {
    m_eTagHasBeenSet = true;
    m_eTag = std::forward<ETagT>(value);
  }
  template <typename ETagT = Aws::String>
  ObjectIdentifier& WithETag(ETagT&& value) {
    SetETag(std::forward<ETagT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If present, the objects are deleted only if its modification times matches
   * the provided <code>Timestamp</code>. </p>  <p>This functionality is only
   * supported for directory buckets.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTime() const { return m_lastModifiedTime; }
  inline bool LastModifiedTimeHasBeenSet() const { return m_lastModifiedTimeHasBeenSet; }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  void SetLastModifiedTime(LastModifiedTimeT&& value) {
    m_lastModifiedTimeHasBeenSet = true;
    m_lastModifiedTime = std::forward<LastModifiedTimeT>(value);
  }
  template <typename LastModifiedTimeT = Aws::Utils::DateTime>
  ObjectIdentifier& WithLastModifiedTime(LastModifiedTimeT&& value) {
    SetLastModifiedTime(std::forward<LastModifiedTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If present, the objects are deleted only if its size matches the provided
   * size in bytes. </p>  <p>This functionality is only supported for directory
   * buckets.</p>
   */
  inline long long GetSize() const { return m_size; }
  inline bool SizeHasBeenSet() const { return m_sizeHasBeenSet; }
  inline void SetSize(long long value) {
    m_sizeHasBeenSet = true;
    m_size = value;
  }
  inline ObjectIdentifier& WithSize(long long value) {
    SetSize(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_key;

  Aws::String m_versionId;

  Aws::String m_eTag;

  Aws::Utils::DateTime m_lastModifiedTime{};

  long long m_size{0};
  bool m_keyHasBeenSet = false;
  bool m_versionIdHasBeenSet = false;
  bool m_eTagHasBeenSet = false;
  bool m_lastModifiedTimeHasBeenSet = false;
  bool m_sizeHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3
}  // namespace Aws
