﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/payment-cryptography/PaymentCryptographyRequest.h>
#include <aws/payment-cryptography/PaymentCryptography_EXPORTS.h>

#include <utility>

namespace Aws {
namespace PaymentCryptography {
namespace Model {

/**
 */
class ListAliasesRequest : public PaymentCryptographyRequest {
 public:
  AWS_PAYMENTCRYPTOGRAPHY_API ListAliasesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAliases"; }

  AWS_PAYMENTCRYPTOGRAPHY_API Aws::String SerializePayload() const override;

  AWS_PAYMENTCRYPTOGRAPHY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The <code>keyARN</code> for which you want to list all aliases.</p>
   */
  inline const Aws::String& GetKeyArn() const { return m_keyArn; }
  inline bool KeyArnHasBeenSet() const { return m_keyArnHasBeenSet; }
  template <typename KeyArnT = Aws::String>
  void SetKeyArn(KeyArnT&& value) {
    m_keyArnHasBeenSet = true;
    m_keyArn = std::forward<KeyArnT>(value);
  }
  template <typename KeyArnT = Aws::String>
  ListAliasesRequest& WithKeyArn(KeyArnT&& value) {
    SetKeyArn(std::forward<KeyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Use this parameter in a subsequent request after you receive a response with
   * truncated results. Set it to the value of <code>NextToken</code> from the
   * truncated response you just received.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAliasesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Use this parameter to specify the maximum number of items to return. When
   * this value is present, Amazon Web Services Payment Cryptography does not return
   * more than the specified number of items, but it might return fewer.</p> <p>This
   * value is optional. If you include a value, it must be between 1 and 100,
   * inclusive. If you do not include a value, it defaults to 50.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAliasesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_keyArn;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_keyArnHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace PaymentCryptography
}  // namespace Aws
